# Техническое устройство: Автоматическая постановка задачи бухгалтерии при прикреплении файлов к договорам

> Основан на Решении «Подписка + Фоновое задание» (ПРД-001)

## Архитектурный подход

Подписка на событие `ПриЗаписи` справочника `ДоговорыКонтрагентовПрисоединенныеФайлы` перехватывает добавление нового файла, проверяет тип договора-владельца и — при совпадении условия — немедленно запускает фоновое задание через `ФоновыеЗадания.Выполнить()`. Фоновое задание работает асинхронно вне сессии пользователя: создаёт задачу через механизм задач платформы с адресацией по роли из `РолиИсполнителей`. Пользователь не блокируется — обработчик подписки возвращает управление мгновенно.

Синхронная часть (обработчик подписки) не выполняет запросов к базе данных — только читает реквизиты переданного объекта. Все обращения к базе вынесены в фоновое задание.

## Объекты метаданных

| Объект | Тип | Действие | Модуль / Форма | Примечание |
|--------|-----|----------|----------------|------------|
| `ПриЗаписиФайлаДоговора` | Подписка на событие | Создать | — | Источник: `Справочник.ДоговорыКонтрагентовПрисоединенныеФайлы`; событие: `ПриЗаписи`; обработчик: `ОбработчикЗадачПоФайламДоговоров.ПриЗаписиФайлаДоговора` |
| `ОбработчикЗадачПоФайламДоговоров` | Общий модуль | Создать | Модуль | Серверный, без контекста (`&НаСервере`); содержит обработчик подписки и экспортную процедуру для фонового задания |
| `РолиИсполнителей` | Справочник | Добавить элемент НСИ | — | Новый непредопределённый элемент «Бухгалтерия договоры реализации»; `ИспользуетсяБезОбъектовАдресации = Истина` |
| `ДоговорыКонтрагентовПрисоединенныеФайлы` | Справочник | Без изменений | — | Точка подключения подписки; реквизит `ВладелецФайла` даёт доступ к `ТипДоговора` |
| `ДоговорыКонтрагентов` | Справочник | Без изменений | — | Читается `ТипДоговора` (Перечисление.ТипыДоговоров) через `ВладелецФайла` |

## Ключевые алгоритмы

---

### Алгоритм 1: Обработчик подписки — проверка условий и запуск фонового задания

**Точка входа:** Подписка `ПриЗаписиФайлаДоговора` → событие `ПриЗаписи` справочника `ДоговорыКонтрагентовПрисоединенныеФайлы`

**Контекст выполнения:** Сервер (общий модуль `ОбработчикЗадачПоФайламДоговоров`, без контекста)

```bsl
// Процедура-обработчик подписки на событие.
// Параметры Источник и Отказ передаются платформой.
Процедура ПриЗаписиФайлаДоговора(Источник, Отказ) Экспорт

    // 1. Защитное условие: реагируем только на новые записи
    Если НЕ Источник.ЭтоНовый() Тогда
        Возврат;
    КонецЕсли;

    // 2. Защитное условие: пропускаем загрузку через обмен данными
    Если Источник.ОбменДанными.Загрузка Тогда
        Возврат;
    КонецЕсли;

    // 3. Читаем тип договора-владельца напрямую из реквизита объекта (без запроса)
    ТипДоговора = Источник.ВладелецФайла.ТипДоговора;

    // 4. Формируем перечень допустимых типов
    ДопустимыеТипы = Новый Массив;
    ДопустимыеТипы.Добавить(Перечисления.ТипыДоговоров.СПокупателем);
    ДопустимыеТипы.Добавить(Перечисления.ТипыДоговоров.СДавальцем);
    ДопустимыеТипы.Добавить(Перечисления.ТипыДоговоров.СДавальцем2_5);

    // 5. Проверяем совпадение
    Если ДопустимыеТипы.Найти(ТипДоговора) = Неопределено Тогда
        Возврат;  // Тип не подходит — задачу не создаём
    КонецЕсли;

    // 6. Формируем параметры для фонового задания
    МассивПараметров = Новый Массив;
    МассивПараметров.Добавить(Источник.Ссылка);          // ссылка на файл
    МассивПараметров.Добавить(Источник.ВладелецФайла);   // ссылка на договор
    МассивПараметров.Добавить(ТекущаяДатаСеанса());      // момент события

    // 7. Запускаем фоновое задание асинхронно и немедленно возвращаем управление
    ФоновыеЗадания.Выполнить(
        "ОбработчикЗадачПоФайламДоговоров.СоздатьЗадачуПоФайлуДоговора",
        МассивПараметров,
        ,                                        // ключ уникальности — не задаём
        "ПРД-001: создание задачи по файлу договора"
    );

КонецПроцедуры
```

**Граничные случаи:**

- **Файл добавлен через обмен данными:** проверка `ОбменДанными.Загрузка` в шаге 2 гарантирует выход без действий — фоновое задание не запускается.
- **Файл обновлён (не новый):** проверка `ЭтоНовый()` в шаге 1 гарантирует, что повторная задача не создаётся при перезаписи существующего файла.
- **Договор-владелец пустой (`ВладелецФайла` = пустая ссылка):** обращение к `ВладелецФайла.ТипДоговора` вернёт пустое значение перечисления; `ДопустимыеТипы.Найти()` вернёт `Неопределено`; задача не создастся.
- **Тип договора не из допустимого перечня:** выход в шаге 5, задача не создаётся.
- **Фоновые задания отключены на сервере:** вызов `ФоновыеЗадания.Выполнить()` завершится с ошибкой; исключение всплывёт в обработчик подписки и будет зафиксировано платформой в журнале регистрации; задача не создаётся; пользователь не уведомляется.

---

### Алгоритм 2: Фоновое задание — создание задачи исполнителя

**Точка входа:** Экспортная процедура `СоздатьЗадачуПоФайлуДоговора` модуля `ОбработчикЗадачПоФайламДоговоров`

**Контекст выполнения:** Сервер (фоновое задание, вне сессии пользователя)

```bsl
// Вызывается фоновым заданием. Параметры передаются через МассивПараметров.
Процедура СоздатьЗадачуПоФайлуДоговора(СсылкаНаФайл, СсылкаНаДоговор, ДатаСобытия) Экспорт

    // 1. Ищем роль исполнителя — см. Алгоритм 3
    РольИсполнителя = НайтиРольБухгалтерияДоговоры();

    Если РольИсполнителя = Неопределено Тогда
        // Роль не найдена — фиксируем в журнале, задача не создаётся
        ЗаписьЖурналаРегистрации(
            "ПРД-001 ОбработчикЗадачПоФайламДоговоров",
            УровеньЖурналаРегистрации.Ошибка,
            ,
            СсылкаНаДоговор,
            "Роль 'Бухгалтерия договоры реализации' не найдена в РолиИсполнителей. Задача не создана."
        );
        Возврат;
    КонецЕсли;

    // 2. Ищем приоритет «Высокий» — см. Алгоритм 4
    Приоритет = НайтиПриоритетВысокий();
    // Если приоритет не найден — создаём задачу с пустым приоритетом (не прерываем выполнение)

    // 3. Создаём задачу через механизм задач платформы
    НоваяЗадача = Задачи.ЗадачаИсполнителя.СоздатьЗадачу();
    НоваяЗадача.Описание       = "К договору добавлены файлы, просьба проверить настройки справочника";
    НоваяЗадача.РольИсполнителя = РольИсполнителя;
    НоваяЗадача.СрокИсполнения  = ДатаСобытия + 86400;  // +1 календарный день (86 400 секунд)
    НоваяЗадача.Приоритет       = Приоритет;             // может быть Неопределено — допустимо

    // 4. Записываем задачу
    НоваяЗадача.Записать();

КонецПроцедуры
```

**Граничные случаи:**

- **Роль «Бухгалтерия договоры реализации» не найдена:** запись ошибки в журнал регистрации, выход без создания задачи. Пользователь не уведомляется — это допустимое ограничение (см. NFR-03 в requirements.md).
- **Приоритет «Высокий» не найден:** задача создаётся с пустым полем `Приоритет`. Ошибка в журнал не пишется — отсутствие приоритета не является критическим сбоем.
- **Ошибка при записи задачи (`НоваяЗадача.Записать()`):** исключение всплывёт, фоновое задание завершится с ошибкой, которая фиксируется в журнале фоновых заданий платформы. Повтора нет.

---

### Алгоритм 3: Поиск роли «Бухгалтерия договоры реализации»

**Контекст:** Вспомогательная функция, вызывается из Алгоритма 2.

```bsl
// Возвращает ссылку на элемент РолиИсполнителей или Неопределено, если не найден.
Функция НайтиРольБухгалтерияДоговоры()

    Запрос = Новый Запрос;
    Запрос.Текст =
        "ВЫБРАТЬ ПЕРВЫЕ 1
        |    РолиИсполнителей.Ссылка
        |ИЗ
        |    Справочник.РолиИсполнителей КАК РолиИсполнителей
        |ГДЕ
        |    РолиИсполнителей.Наименование = &Наименование
        |    И НЕ РолиИсполнителей.ПометкаУдаления";

    Запрос.УстановитьПараметр("Наименование", "Бухгалтерия договоры реализации");

    Результат = Запрос.Выполнить();
    Если Результат.Пустой() Тогда
        Возврат Неопределено;
    КонецЕсли;

    Возврат Результат.Выгрузить()[0].Ссылка;

КонецФункции
```

---

### Алгоритм 4: Поиск приоритета «Высокий»

**Контекст:** Вспомогательная функция, вызывается из Алгоритма 2.

```bsl
// Возвращает ссылку на элемент Справочник.Приоритеты с наименованием «Высокий»
// или Неопределено, если не найден.
Функция НайтиПриоритетВысокий()

    Запрос = Новый Запрос;
    Запрос.Текст =
        "ВЫБРАТЬ ПЕРВЫЕ 1
        |    Приоритеты.Ссылка
        |ИЗ
        |    Справочник.Приоритеты КАК Приоритеты
        |ГДЕ
        |    Приоритеты.Наименование = &Наименование
        |    И НЕ Приоритеты.ПометкаУдаления
        |УПОРЯДОЧИТЬ ПО
        |    Приоритеты.РеквизитДопУпорядочивания УБЫВ";

    Запрос.УстановитьПараметр("Наименование", "Высокий");

    Результат = Запрос.Выполнить();
    Если Результат.Пустой() Тогда
        Возврат Неопределено;
    КонецЕсли;

    Возврат Результат.Выгрузить()[0].Ссылка;

КонецФункции
```

## Риски реализации

| Риск | Вероятность | Митигация |
|------|-------------|-----------|
| Фоновые задания отключены на сервере приложений | Низкая | Проверить при развёртывании: `ФоновыеЗадания.Выполнить()` должен быть доступен; задокументировать как предусловие |
| Массовое создание фоновых заданий при пакетном импорте файлов через интерфейс | Низкая | Защита через `ОбменДанными.Загрузка` покрывает обмены; при ручном массовом добавлении нагрузка приемлема |
| Изменение наименования элементов «Высокий» или «Бухгалтерия договоры реализации» | Низкая | Поиск по наименованию; при переименовании задача перестанет создаваться с нужным приоритетом/адресатом; рекомендуется кэшировать ссылки в константах после стабилизации |
| Обновление конфигурации меняет состав `Перечисление.ТипыДоговоров` | Низкая | Проверять перечень значений при каждом мажорном обновлении конфигурации |
| Задача `Задачи.ЗадачаИсполнителя` не имеет реквизита `Приоритет` в данной конфигурации | Средняя | Уточнить у разработчика наличие реквизита `Приоритет` на объекте `Задача.ЗадачаИсполнителя`; при отсутствии — исключить заполнение из Алгоритма 2 |

## Зависимости

**Требует наличия:**
- Механизм задач платформы (`Задачи.ЗадачаИсполнителя`) — должен быть включён и настроен в конфигурации
- Фоновые задания — должны быть разрешены на сервере приложений
- Элемент «Бухгалтерия договоры реализации» в `РолиИсполнителей` — создаётся в рамках данной задачи (фаза 1)
- Элемент «Высокий» в `Справочник.Приоритеты` — должен существовать в базе данных

**Влияет на:**
- Нагрузку на очередь фоновых заданий — минимальная при штатном режиме добавления файлов
